# say.py
"""
========================================
💬 SAYモジュール - テキスト表示システム（Actor統合版）
========================================

pygame-zeroでキャラクターがセリフを言ったり、テキストを表示するためのモジュールです。
色付きテキスト、フォント指定、位置調整、時間制限表示などの高機能を提供します。
Actorクラスを拡張し、actor.say()で直接セリフが言えるようになりました！

📋 基本的な使用方法:
    from say import text_display
    
    def draw():
        text_display.draw(screen)  # draw関数で呼び出し（必須）
    
    # 新しい書き方（推奨）
    boon.say("こんにちは！", 3)              # Actorから直接セリフ
    if boon.is_talking():                   # 話し中かチェック
        print("boonは話し中です")
    boon.clear_text()                       # セリフをクリア
    
    # 従来の書き方（互換性のため残存）
    text_display.say(boon, "こんにちは！", 3)

========================================
🎯 Actorクラスの新機能（推奨）
========================================

🔸 actor.say(text, seconds=None, **kwargs)
    説明: このActorがテキストを表示する
    パラメータ:
        text: 表示するテキスト（色指定タグ付き可能）
        seconds: 表示する秒数（Noneの場合は永続表示）
        size: フォントサイズ（デフォルト40）
        color: デフォルトのテキスト色（デフォルト"black"）
        x_offset: このテキスト用のX座標オフセット
        y_offset: このテキスト用のY座標オフセット
        fontname: フォント名（日本語表示用）
        anchor: テキストの座標基準点（デフォルト"center"）
    
    例:
        boon.say("こんにちは！", 3)
        boon.say("大きな文字", 2, size=60, color="red")
        boon.say("<red>赤い</red><blue>青い</blue>テキスト", 3)
        boon.say("上に表示", 2, y_offset=-100)

🔸 actor.is_talking()
    説明: このActorが現在話し中かどうかを返す
    戻り値: True=話し中、False=話していない
    例: 
        if boon.is_talking():
            print("boonは話し中")
        
        if not enemy.is_talking():
            enemy.say("攻撃するぞ！", 2)

🔸 actor.clear_text()
    説明: このActorのテキスト表示をクリアする
    例: boon.clear_text()

========================================
📂 text_displayプロパティ
========================================

🔹 text_display.x_offset (または text_display.x)
    型: 数値
    説明: テキストのX座標オフセット（デフォルト値）
    例: text_display.x_offset = 10  # 右に10ピクセルずらす

🔹 text_display.y_offset (または text_display.y)
    型: 数値
    説明: テキストのY座標オフセット（デフォルト値）
    例: text_display.y_offset = -80  # 上に80ピクセルずらす

========================================
🛠️ text_displayメソッド（従来版・互換性用）
========================================

🔸 text_display.say(obj, text, seconds=None, **kwargs)
    説明: オブジェクトがテキストを表示する（従来版）
    例: text_display.say(player, "こんにちは！", 3)

🔸 text_display.clear(obj)
    説明: 指定されたオブジェクトのテキスト表示をクリアする
    例: text_display.clear(player)

🔸 text_display.is_talking(obj)
    説明: 指定されたオブジェクトが現在話し中かどうかを返す
    例: if text_display.is_talking(player): print("話し中")

🔸 text_display.draw(screen)
    説明: テキストを画面に描画する（draw関数で呼び出し必須）
    例: text_display.draw(screen)

🔸 text_display.update(dt)
    説明: テキストの時間管理を更新する（通常は自動実行）
    例: text_display.update(dt)  # 自動更新が無効な場合のみ必要

========================================
🎨 色指定タグの使い方
========================================

🔸 基本的な色名
    <red>赤いテキスト</red>
    <blue>青いテキスト</blue>
    <green>緑のテキスト</green>
    <yellow>黄色のテキスト</yellow>
    <white>白いテキスト</white>
    <black>黒いテキスト</black>
    <orange>オレンジのテキスト</orange>
    <purple>紫のテキスト</purple>
    <pink>ピンクのテキスト</pink>
    <brown>茶色のテキスト</brown>
    <gray>灰色のテキスト</gray>

🔸 HEXコード指定
    <#FF0000>赤いテキスト</>
    <#00FF00>緑のテキスト</>
    <#0000FF>青いテキスト</>

🔸 RGB値指定
    <255,0,0>赤いテキスト</>
    <0,255,0>緑のテキスト</>
    <0,0,255>青いテキスト</>

🔸 複数色の組み合わせ
    "こんにちは<red>赤い</red>世界<blue>青い</blue>空"

========================================
📍 anchor（座標基準点）の設定
========================================

🔸 上部
    "topleft"     - 左上
    "topcenter"   - 中央上  
    "topright"    - 右上

🔸 中央
    "centerleft"  - 左中央
    "center"      - 中央（デフォルト）
    "centerright" - 右中央

🔸 下部
    "bottomleft"  - 左下
    "bottomcenter"- 中央下
    "bottomright" - 右下

========================================
🎪 実践例（新しいActor統合版）
========================================

# 基本的なセリフシステム
def on_key_down(key):
    if key == keys.SPACE:
        npc.say("こんにちは、冒険者よ！", 3)

# 会話システム
conversation = [
    "はじめまして！",
    "<blue>青い</blue>魔法を覚えたよ",
    "また会いましょう！"
]
current_line = 0

def next_conversation():
    global current_line
    if current_line < len(conversation):
        npc.say(conversation[current_line], 3)
        current_line += 1

# 状況に応じたメッセージ
def update_game():
    if player.health < 20:
        if not player.is_talking():
            player.say("<red>体力が少ない！</red>", 2, size=50)
    
    if enemy.colliderect(player):
        enemy.say("<purple>発見したぞ！</purple>", 1.5, anchor="topcenter")

# 衝突時のセリフ
def ball_move():
    if ball.collide_pixel(boon):
        boon.say("Ouch!", 2, size=70, color="red", y_offset=-70)

# 移動制御（話し中は動かない）
def boon_move():
    if not boon.is_talking():
        boon.x += boon_speed
        # 移動処理...

========================================
🔧 必須の設定
========================================

draw関数での呼び出しが必要です:

def draw():
    screen.fill("white")
    player.draw()
    enemy.draw()
    text_display.draw(screen)  # ← 必須！

========================================
💡 ヒントとコツ
========================================

🔸 新しい書き方を使おう
    actor.say() の方が直感的で分かりやすいです
    
🔸 話し中の確認
    actor.is_talking() でセリフ中の動作制御ができます
    
🔸 永続表示
    seconds=None にすると消えないテキストになります
    
🔸 色とサイズで重要度を表現
    重要な情報は大きく、色付きで表示しましょう

========================================
"""

# モジュールレベルの変数
_registered = False  # 更新処理が登録されたかどうか
_last_time = 0  # 前回の更新時間

import re

class TextDisplay:
    def __init__(self):
        self.texts = {}  # テキスト情報を保存する辞書
        self.x_offset = 0  # デフォルトのX座標オフセット
        self.y_offset = -50  # デフォルトのY座標オフセット（上に50ピクセル）
        self.talking_objects = set()  # 話しているオブジェクトのセット
        
    def say(self, obj, text, seconds=None, size=40, color="black", x_offset=None, y_offset=None, fontname=None, anchor="center"):
        """
        オブジェクトがテキストを表示する
        obj: 表示位置の基準となるオブジェクト（Actor等）
        text: 表示するテキスト（色指定タグ付き可能）
        seconds: 表示する秒数（Noneの場合は永続表示）
        size: フォントサイズ
        color: デフォルトのテキスト色
        x_offset: このテキスト用のX座標オフセット（指定しない場合はデフォルト値を使用）
        y_offset: このテキスト用のY座標オフセット（指定しない場合はデフォルト値を使用）
        fontname: フォント名（日本語表示用）
        anchor: テキストの座標基準点（デフォルト: "center"）
        
        anchor設定:
        "topleft" - 左上        "topcenter" - 中央上       "topright" - 右上
        "centerleft" - 左中央   "center" - 中央           "centerright" - 右中央  
        "bottomleft" - 左下     "bottomcenter" - 中央下    "bottomright" - 右下
        
        色指定の記法例:
        "<red>こんにちは</red><blue>！</blue>"
        "<#FF0000>赤いテキスト</>"
        "普通の<green>緑の</green>テキスト"
        """
        # 自動更新の登録（初回のみ）
        global _registered
        if not _registered:
            try:
                import pgzero.clock
                # 引数なしのコールバックを登録
                pgzero.clock.schedule_interval(self._auto_update_wrapper, 1/60)
                _registered = True
            except (ImportError, AttributeError):
                # 初期化前の場合は何もしない（後でユーザーがupdate関数を呼ぶ）
                pass
                
        # テキストを解析して色付き文字列のリストに変換
        text_parts = self._parse_colored_text(text, color)
                
        # テキスト情報を保存
        self.texts[obj] = {
            "text_parts": text_parts,  # 色付き文字列のリスト
            "timer": seconds,  # Noneの場合は永続表示を意味する
            "size": size,
            "default_color": color,
            "x_offset": x_offset if x_offset is not None else self.x_offset,
            "y_offset": y_offset if y_offset is not None else self.y_offset,
            "fontname": fontname,  # フォント名を追加
            "anchor": anchor  # 座標基準点を追加
        }
        
        # このオブジェクトを話し中に追加
        self.talking_objects.add(obj)
    
    def _parse_colored_text(self, text, default_color):
        """
        色指定タグ付きテキストを解析して、(文字, 色)のリストに変換
        """
        # 色指定タグのパターン: <color>text</color> または <color>text</>
        pattern = r'<([^>]+)>(.*?)(?:</[^>]*>|</>)'
        
        parts = []
        last_end = 0
        
        for match in re.finditer(pattern, text):
            # タグの前の普通のテキスト
            if match.start() > last_end:
                normal_text = text[last_end:match.start()]
                if normal_text:
                    parts.append((normal_text, default_color))
            
            # 色指定されたテキスト
            color_spec = match.group(1)
            colored_text = match.group(2)
            
            # 色名の変換（必要に応じて拡張）
            color = self._convert_color_name(color_spec)
            parts.append((colored_text, color))
            
            last_end = match.end()
        
        # 最後の普通のテキスト
        if last_end < len(text):
            remaining_text = text[last_end:]
            if remaining_text:
                parts.append((remaining_text, default_color))
        
        # タグが一つもない場合は全体をデフォルト色で
        if not parts:
            parts.append((text, default_color))
            
        return parts
    
    def _convert_color_name(self, color_spec):
        """
        色名やHEXコードを適切な形式に変換
        """
        # 基本的な色名のマッピング
        color_map = {
            'red': 'red',
            'blue': 'blue', 
            'green': 'green',
            'yellow': 'yellow',
            'white': 'white',
            'black': 'black',
            'orange': 'orange',
            'purple': 'purple',
            'pink': 'pink',
            'brown': 'brown',
            'gray': 'gray',
            'grey': 'gray'
        }
        
        # 色名の場合
        if color_spec.lower() in color_map:
            return color_map[color_spec.lower()]
        
        # HEXコード（#で始まる）の場合はそのまま返す
        if color_spec.startswith('#'):
            return color_spec
            
        # RGB値（255,0,0形式）の場合
        if ',' in color_spec:
            try:
                rgb = tuple(map(int, color_spec.split(',')))
                return rgb
            except:
                pass
        
        # 不明な場合はそのまま返す（Pygameが解釈できるかもしれない）
        return color_spec
    
    def clear(self, obj):
        """
        指定されたオブジェクトのテキスト表示をクリアする
        """
        if obj in self.texts:
            del self.texts[obj]
            if obj in self.talking_objects:
                self.talking_objects.remove(obj)
    
    def is_talking(self, obj):
        """
        指定されたオブジェクトが現在話し中かどうかを返す
        """
        return obj in self.talking_objects
    
    def _auto_update_wrapper(self):
        """
        スケジューラに登録するラッパー関数（引数なし）
        """
        import time
        global _last_time
        current_time = time.time()
        if _last_time == 0:
            dt = 1/60  # 初回は仮の値
        else:
            dt = current_time - _last_time
        _last_time = current_time
        
        # 実際の更新関数を呼び出す
        self._auto_update(dt)
    
    def _auto_update(self, dt):
        """
        自動更新処理（内部使用）
        dt: 経過時間（秒）
        """
        self.update(dt)
    
    def update(self, dt):
        """
        update関数から呼び出す（自動更新が機能しない場合用）
        dt: 経過時間（秒）
        """
        # 各テキストのタイマーを更新
        to_remove = []
        for obj, info in self.texts.items():
            # timerがNoneの場合は永続表示なので、時間を減らさない
            if info["timer"] is not None:
                info["timer"] -= dt
                if info["timer"] <= 0:
                    to_remove.append(obj)
        
        # 時間切れのテキストを削除
        for obj in to_remove:
            del self.texts[obj]
            # このオブジェクトを話し中から削除
            if obj in self.talking_objects:
                self.talking_objects.remove(obj)
    
    def draw(self, screen):
        """
        draw関数から呼び出す
        screen: Pygame Zeroのscreen
        """
        for obj, info in self.texts.items():
            # 各テキスト専用のオフセットを取得
            x_offset = info.get("x_offset", self.x_offset)
            y_offset = info.get("y_offset", self.y_offset)
            fontname = info.get("fontname", None)
            anchor = info.get("anchor", "center")
            
            # ベース座標を計算
            base_x = obj.x + x_offset
            base_y = obj.y + y_offset
            
            # 全体テキストを連結して幅を計算
            full_text = ""
            for text_part, _ in info["text_parts"]:
                full_text += text_part
            total_width = self._calculate_text_width(screen, full_text, info["size"], fontname)
            text_height = info["size"]
            
            # anchor設定に基づいてテキスト描画位置を調整
            anchor_x, anchor_y = self._get_anchor_position(base_x, base_y, total_width, text_height, anchor)
            
            # 各anchorに応じた開始X座標を計算
            if "left" in anchor:
                start_x = anchor_x  # 左寄せの場合、anchor_xから開始
            elif "right" in anchor:
                start_x = anchor_x - total_width  # 右寄せの場合、右端から逆算
            else:  # center
                start_x = anchor_x - total_width // 2  # 中央の場合、中心から左右に分ける
            
            current_x = start_x
            
            # 各パーツを順番に描画
            for text_part, color in info["text_parts"]:
                if text_part:
                    part_width = self._calculate_text_width(screen, text_part, info["size"], fontname)
                    
                    # 描画
                    if fontname:
                        screen.draw.text(
                            text_part,
                            center=(current_x + part_width // 2, anchor_y),
                            fontsize=info["size"],
                            color=color,
                            fontname=fontname
                        )
                    else:
                        screen.draw.text(
                            text_part,
                            center=(current_x + part_width // 2, anchor_y),
                            fontsize=info["size"],
                            color=color
                        )
                    
                    current_x += part_width
    
    def _get_anchor_position(self, base_x, base_y, width, height, anchor):
        """
        anchor設定に基づいて実際の座標を計算
        """
        # X座標の調整
        if "left" in anchor:
            anchor_x = base_x + width // 30  # テキスト左端がbase_x位置に近づくように
        elif "right" in anchor:
            anchor_x = base_x - width // 30  # テキスト右端がbase_x位置に近づくように
        else:  # center
            anchor_x = base_x  # テキスト中央がbase_x位置
        
        # Y座標の調整
        if anchor.startswith("top"):
            anchor_y = base_y + height // 2  # テキスト上端がbase_y位置に来るように
        elif anchor.startswith("bottom"):
            anchor_y = base_y - height // 2  # テキスト下端がbase_y位置に来るように
        else:  # center
            anchor_y = base_y  # テキスト中央がbase_y位置
        
        return anchor_x, anchor_y
    
    def _calculate_text_width(self, screen, text, size, fontname=None):
        """
        テキストの幅を計算（フォント対応版）
        """
        try:
            import pygame
            # フォント指定がある場合とない場合を分ける
            if fontname:
                # フォントファイルの場合
                if fontname.endswith('.ttf') or fontname.endswith('.otf'):
                    try:
                        font = pygame.font.Font(f"fonts/{fontname}", size)
                    except:
                        font = pygame.font.SysFont("arial", size)  # フォールバック
                else:
                    # システムフォント名の場合
                    font = pygame.font.SysFont(fontname, size)
            else:
                font = pygame.font.Font(None, size)
            
            text_surface = font.render(text, True, (0, 0, 0))
            return text_surface.get_width()
        except:
            # フォールバック：近似計算
            return len(text) * int(size * 0.6)
    
    def _calculate_total_width(self, screen, text_parts, size, fontname=None):
        """
        全体のテキスト幅を計算（フォント対応版）
        """
        total_width = 0
        for text_part, _ in text_parts:
            total_width += self._calculate_text_width(screen, text_part, size, fontname)
        return total_width
    
    # x_offsetプロパティのゲッター
    @property
    def x(self):
        return self.x_offset
    
    # x_offsetプロパティのセッター
    @x.setter
    def x(self, value):
        self.x_offset = value
    
    # y_offsetプロパティのゲッター
    @property
    def y(self):
        return self.y_offset
    
    # y_offsetプロパティのセッター
    @y.setter
    def y(self, value):
        self.y_offset = value

# グローバルなインスタンス
text_display = TextDisplay()

# Actorクラスの拡張
def _enhance_actor():
    """
    Actorクラスに say, is_talking, clear_text メソッドを追加する
    """
    try:
        # pgzero.actorをインポート
        from pgzero.actor import Actor
        
        # sayメソッドを追加
        def say(self, text, seconds=None, size=40, color="black", x_offset=None, y_offset=None, fontname=None, anchor="center"):
            """
            このActorがテキストを表示する
            使用例: boon.say("Hello!", 3, size=50, color="blue")
            """
            text_display.say(self, text, seconds, size, color, x_offset, y_offset, fontname, anchor)
        
        # is_talkingメソッドを追加
        def is_talking(self):
            """
            このActorが現在話し中かどうかを返す
            使用例: if boon.is_talking():
            """
            return text_display.is_talking(self)
        
        # clear_textメソッドを追加
        def clear_text(self):
            """
            このActorのテキスト表示をクリアする
            使用例: boon.clear_text()
            """
            text_display.clear(self)
        
        # Actorクラスにメソッドを追加
        Actor.say = say
        Actor.is_talking = is_talking
        Actor.clear_text = clear_text
        
    except ImportError:
        # pgzero.actorが利用できない場合は何もしない
        pass

# pgzhelperとの互換性を確保
def _enhance_pgzhelper_actor():
    """
    pgzhelperのActorクラスにもメソッドを追加する
    """
    try:
        # pgzhelperをインポート
        import pgzhelper
        
        # pgzhelperにActorクラスがあるかチェック
        if hasattr(pgzhelper, 'Actor'):
            Actor = pgzhelper.Actor
            
            # sayメソッドを追加
            def say(self, text, seconds=None, size=40, color="black", x_offset=None, y_offset=None, fontname=None, anchor="center"):
                """
                このActorがテキストを表示する
                使用例: boon.say("Hello!", 3, size=50, color="blue")
                """
                text_display.say(self, text, seconds, size, color, x_offset, y_offset, fontname, anchor)
            
            # is_talkingメソッドを追加
            def is_talking(self):
                """
                このActorが現在話し中かどうかを返す
                使用例: if boon.is_talking():
                """
                return text_display.is_talking(self)
            
            # clear_textメソッドを追加
            def clear_text(self):
                """
                このActorのテキスト表示をクリアする
                使用例: boon.clear_text()
                """
                text_display.clear(self)
            
            # Actorクラスにメソッドを追加
            Actor.say = say
            Actor.is_talking = is_talking
            Actor.clear_text = clear_text
            
    except ImportError:
        # pgzhelperが利用できない場合は何もしない
        pass

# モジュール読み込み時に自動的にActorクラスを拡張
_enhance_actor()
_enhance_pgzhelper_actor()

# 後方互換性のためのエイリアス
# 既存のコードが動作するように
def draw_texts(screen):
    """
    後方互換性のための関数
    text_display.draw(screen) の代わりに使える
    """
    text_display.draw(screen)

def update_texts(dt):
    """
    後方互換性のための関数
    text_display.update(dt) の代わりに使える
    """
    text_display.update(dt)
